/*
 * Copyright (c) 2016, 2018, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package jdk.javadoc.internal.doclets.toolkit;

import java.text.MessageFormat;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * Access to the localizable resources used by a doclet.
 * The resources are split across two resource bundles:
 * one that contains format-neutral strings common to
 * all supported formats, and one that contains strings
 * specific to the selected doclet, such as the standard
 * HTML doclet.
 */
public class Resources {
    public final String annotationTypeSummary;
    public final String annotationTypeTableSummary;
    public final String classSummary;
    public final String classTableSummary;
    private final BaseConfiguration configuration;
    private final String commonBundleName;
    private final String docletBundleName;
    public final String enumSummary;
    public final String enumTableSummary;
    public final String errorSummary;
    public final String errorTableSummary;
    public final String exceptionSummary;
    public final String exceptionTableSummary;
    public final String interfaceSummary;
    public final String interfaceTableSummary;
    public final String packageSummary;
    public final String packageTableSummary;

    protected ResourceBundle commonBundle;
    protected ResourceBundle docletBundle;

    /**
     * Creates a {@code Resources} to provide access the resource
     * bundles used by a doclet.
     *
     * @param configuration the configuration for the doclet,
     *  to provide access the locale to be used when accessing the
     *  names resource bundles.
     * @param commonBundleName the name of the bundle containing the strings
     *  common to all output formats
     * @param docletBundleName the name of the bundle containing the strings
     *  specific to a particular format
     */
    public Resources(BaseConfiguration configuration, String commonBundleName, String docletBundleName) {
        this.configuration = configuration;
        this.commonBundleName = commonBundleName;
        this.docletBundleName = docletBundleName;
        this.annotationTypeSummary = getText("doclet.Annotation_Types_Summary");
        this.annotationTypeTableSummary = getText("doclet.Member_Table_Summary",
                this.annotationTypeSummary, getText("doclet.annotationtypes"));
        this.classSummary = getText("doclet.Class_Summary");
        this.classTableSummary = getText("doclet.Member_Table_Summary",
                this.classSummary, getText("doclet.classes"));
        this.enumSummary = getText("doclet.Enum_Summary");
        this.enumTableSummary = getText("doclet.Member_Table_Summary",
                this.enumSummary, getText("doclet.enums"));
        this.errorSummary = getText("doclet.Error_Summary");
        this.errorTableSummary = getText("doclet.Member_Table_Summary",
                this.errorSummary, getText("doclet.errors"));
        this.exceptionSummary = getText("doclet.Exception_Summary");
        this.exceptionTableSummary = getText("doclet.Member_Table_Summary",
                this.exceptionSummary, getText("doclet.exceptions"));
        this.interfaceSummary = getText("doclet.Interface_Summary");
        this.interfaceTableSummary = getText("doclet.Member_Table_Summary",
                this.interfaceSummary, getText("doclet.interfaces"));
        this.packageSummary = getText("doclet.Package_Summary");
        this.packageTableSummary = getText("doclet.Member_Table_Summary",
                this.packageSummary, getText("doclet.packages"));
    }

    /**
     * Gets the string for the given key from one of the doclet's
     * resource bundles.
     *
     * The more specific bundle is checked first;
     * if it is not there, the common bundle is then checked.
     *
     * @param key the key for the desired string
     * @return the string for the given key
     * @throws MissingResourceException if the key is not found in either
     *  bundle.
     */
    public String getText(String key) throws MissingResourceException {
        initBundles();

        if (docletBundle.containsKey(key))
            return docletBundle.getString(key);

        return commonBundle.getString(key);
    }
    /**
     * Gets the string for the given key from one of the doclet's
     * resource bundles, substituting additional arguments into
     * into the resulting string with {@link MessageFormat#format}.
     *
     * The more specific bundle is checked first;
     * if it is not there, the common bundle is then checked.
     *
     * @param key the key for the desired string
     * @param args values to be substituted into the resulting string
     * @return the string for the given key
     * @throws MissingResourceException if the key is not found in either
     *  bundle.
     */
    public String getText(String key, Object... args) throws MissingResourceException {
        return MessageFormat.format(getText(key), args);
    }

    /**
     * Lazily initializes the bundles. This is (currently) necessary because
     * this object may be created before the locale to be used is known.
     */
    protected void initBundles() {
        if (commonBundle == null) {
            Locale locale = configuration.getLocale();
            this.commonBundle = ResourceBundle.getBundle(commonBundleName, locale);
            this.docletBundle = ResourceBundle.getBundle(docletBundleName, locale);
        }
    }
}
